<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Intervention\Image\Facades\Image;
use Spatie\Permission\Models\Role;
use DB;
use Hash;
use Illuminate\Support\Arr;

class UserController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $roles = Role::pluck('name','name')->all();
        $data = User::orderBy('id','DESC')->paginate(5);
        return view('panel.users.index',compact('data','roles'))->with('i', ($request->input('page', 1) - 1) * 5);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $roles = Role::pluck('name','name')->all();
        return view('users.create',compact('roles'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */

    public function store(Request $request)
    {


        $contact_check = MobileNumberValidator($request->country_code,  strval(intval($request->contact)),"null");

        if($contact_check === true)
        {
            return redirect()->back()->with('error','Dubpicate contact found');
        }

        $this->validate($request, [
            'name' => 'required',
            'email' => 'required|email|unique:users,email',
            'username' => 'required|string|unique:users,username',
            'password' => 'required|same:confirm-password',
            'image' => 'image|mimes:jpg,png,jpeg,gif,svg|max:10100',
            'roles' =>   'required'
        ]);

        $input = $request->all();

        $input['password'] = Hash::make($input['password']);

        $input['contact'] = strval(intval($request->contact ?? 0));

        $user = User::create($input);

        $user->assignRole($request->input('roles'));

        if($request->hasFile('image'))
        {

            $image = $request->file('image');
            $input['temp_images'] = time().'.'.$image->getClientOriginalExtension();

            $destinationPath = storage_path( 'app/public/users/'.$user['id'].'/');

            onUploadImage($destinationPath,$input,$image,null);

            \Illuminate\Support\Facades\DB::table('users')->where('id', $user['id'])->update(['image'=>$input['temp_images']]);

        }

        return redirect()->route('panel.users.index')->with('success','User created successfully');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $user = User::find($id);
        return view('users.show',compact('user'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {


        $user = User::find($id);

        $roles = Role::pluck('name','name')->all();

        $userRole = $user->roles->pluck('name','name')->all();

        $data['user'] = $user;
        $data['role'] = $roles;
        $data['userRole'] = $userRole;

        return response($data,200);

    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
//        dd($request);


        $contact_check = MobileNumberValidator($request->country_code,  strval(intval($request->contact)),$id);

        if($contact_check === true)
        {
            return redirect()->back()->with('error','Dubpicate Contact found');
        }

        $validator = Validator::make($request->all(), [
            'name' => 'required',
            'email' => 'required|email|unique:users,email,'.$id,
            'username' => 'string|unique:users,username,'.$id,
            'password' => 'same:confirm-password',
            'image' => 'image|mimes:jpg,png,jpeg,gif,svg|max:10100',
            'roles' => 'required'
        ]);

        if ($validator->fails())
        {
            foreach($validator->errors()->all() as $list)
            {
                toastr()->error($list);
            }

            return redirect()->back();
        }


        $input = $request->all();

        if(!empty($input['password'])){
            $input['password'] = Hash::make($input['password']);
        }else{
            $input = Arr::except($input,array('password'));
        }

//      dd($input);
        $user = User::find($id);
        $user->update($input);

        if($request->hasFile('image'))
        {

            $image = $request->file('image');

            $input['temp_images'] = time().'.'.$image->getClientOriginalExtension();

            $destinationPath = storage_path( 'app/public/users/'.$user['id'].'/');

            $delete_path = 'public/users/'.$user['id'].'/'.$user['image'];

            onUploadImage($destinationPath,$input,$image,$delete_path);

            \Illuminate\Support\Facades\DB::table('users')->where('id', $user['id'])->update(['image'=>$input['temp_images']]);

        }

        DB::table('model_has_roles')->where('model_id',$id)->delete();

        $user->assignRole($request->input('roles'));

        return redirect()->route('panel.users.index')
            ->with('success','User updated successfully');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $user = User::find($id);

        $delete_path = 'public/users/'.$user['id']."/";

        // validation for deleting image so it can be upload
        if(Storage::exists($delete_path)){

            Storage::deleteDirectory($delete_path);

        }

        User::find($id)->delete();





        return redirect()->route('panel.users.index')
            ->with('success','User deleted successfully');
    }

    public function GetLogout(Request $request)
    {
        Auth::logout();
        return redirect()->route('login');
    }


    public function Profile()
    {
        return view('panel.users.profile');
    }

    public function ProfileUpdate(Request $request)
    {

        $contact_check = MobileNumberValidator($request->country_code,  strval(intval($request->contact)),Auth::User()->id);

        if($contact_check === true)
        {
            return redirect()->back()->with('error','Dubpicate Contact found');
        }

        $validator = Validator::make($request->all(), [
            'name' => 'required',
            'email' => 'required|email|unique:users,email,'.Auth::User()->id,
            'username' => 'string|unique:users,username,'.Auth::User()->id,
            'password' => 'same:confirm-password',
            'image' => 'image|mimes:jpg,png,jpeg,gif,svg|max:10100',

        ]);

        if ($validator->fails())
        {
            foreach($validator->errors()->all() as $list)
            {
                toastr()->error($list);
            }

            return redirect()->back();
        }


        $input = $request->all();

        if(!empty($input['password'])){
            $input['password'] = Hash::make($input['password']);
        }else{
            $input = Arr::except($input,array('password'));
        }

//      dd($input);
        $user = User::find(Auth::User()->id);
        $user->update($input);

        if($request->hasFile('image'))
        {

            $image = $request->file('image');

            $input['temp_images'] = time().'.'.$image->getClientOriginalExtension();

            $destinationPath = storage_path( 'app/public/users/'.$user['id'].'/');

            $delete_path = 'public/users/'.$user['id'].'/'.$user['image'];

            onUploadImage($destinationPath,$input,$image,$delete_path);

            \Illuminate\Support\Facades\DB::table('users')->where('id', $user['id'])->update(['image'=>$input['temp_images']]);

        }

        return redirect()->back()->with('success','Profile updated successfully');
    }

}
